#ifndef LLF_C2_H
#define LLF_C2_H

/*
* All the includes that are needed for code using this module to
* compile correctly should be #included here.
*/

#include "CE2_public.h"
#include "CE2_C2.h"

#ifdef __cplusplus
extern "C"
{
#endif

/*
*  Object name     :  LLF_C2.h
*  State           :  %state%
*  Creation date   :  28.03.2007
*  Last modified   :  %modify_time%
*/
/** @file
* \brief contains declarations of low level functions for C2.
*
*  \version LLF_C2.h#1:cinc:1
* \author Yermalayeu Ihar
* \remarks Copyright (C) 2007 by Discretix Technologies Ltd. All rights reserved.
*/

/************************ Defines *****************************/
/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Structs *****************************/
/************************ Public Variables ********************/
/************************ Public Functions ********************/

/**
****************************************************************
* Function Name: 
*  LLF_C2
*
*  @param Key_ptr [in] - A pointer to a buffer of the key to 
*                        operate the C2Cipher operations.
*  @param EncryptDecryptFlag [in] - A flag that determines what 
*                                   operation is performed - 
*                                   encrypting the data or 
*                                   decrypting the data.
*  @param OperationMode [in] - The operation mode of the C2Cipher
*                              (ECB or C-CBC);
*  @param DataIn_ptr [in] - A pointer to a buffer of the input 
*                           data that the C2Cipher operation will 
*                           be executed on.
*  @param DataInSize [in] - The size of the input data block in bytes
*                           Must be a multiple of 8 bytes.
*  @param DataOut_ptr [out] - A pointer to the output buffer that 
*                             shall contain the results.
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*
* \brief \b 
* Description:
*  This function provides a C2 function for processing data.
*
*  \b 
* Algorithm:
*  -# Load the secret constant if need.
*  -# Choose sub-algorithm corresponding to given mode.
***************************************************************/
CE2Error_t LLF_C2(
  CE2_C2Cipher_Key_t           Key_ptr,            /*in*/     
  CE2_C2Cipher_EncryptMode_t   EncryptDecryptFlag, /*in*/
  CE2_C2Cipher_OperationMode_t OperationMode ,     /*in*/ 
  DxUint8_t                    *DataIn_ptr,        /*in*/ 
  DxUint32_t                   DataInSize,         /*in*/ 
  DxUint8_t                    *DataOut_ptr );     /*in/out*/

/**
****************************************************************
* Function Name: 
*  LLF_C2_HASH
*
*  @param[in] DataIn_ptr - A pointer to the buffer that stores the data 
*                          to be hashed .
*  @param[in] DataInSize - The size of the data to be hashed in bytes. 
*  @param[out] HashResultBuff - A pointer to the target buffer where the 
*                               C2 HASH result stored in the context is 
*                               loaded to.
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*
* \brief \b 
* Description:
*  This function performs all C2 HASH operations on one buffer of data.
*
*  \b 
* Algorithm:
*  -# Load the secret constant if need.
*  -# Pad input data by '10...0' to make the its total length a multiple of 64 bits.
*  -# Set 56 least significant bits of 64-bit value 'h0' as initial key value.
*  -# Process the main data blocks 
*     a. Executing the C2 ECB encrypt operation for i-th data block 
*     b. Performs XOR of encrypt output with i-th data block 
*     c. Set 56 least significant bits of 'HashResultBuff' as a next key value. 
*  -# Process (a-c) the last data block. Hash is result of this operation.
***************************************************************/
CE2Error_t LLF_C2_HASH(DxUint8_t              *DataIn_ptr,
                       DxUint32_t             DataSize,
                       CE2_C2HASH_Result_t    HashResultBuff);

/**
****************************************************************
* Function Name: 
*  LLF_C2_OneWayFunc
*
*  @param[in] Data1_ptr - The pointer to the buffer of the input Data1 
*                         used as encrypting key. The pointer need to 
*                         be aligned to 32 bits.
*  @param[in] Data2 - The pointer to the 64-bits buffer of the input 
*                     Data2. The pointer need to be aligned to 32 bits.
*  @param[in/out] DataOut_ptr - The pointer to the 64-bits buffer for 
*                               output data. The pointer need to be 
*                               aligned to 32 bits. 
*
* @returns \b
*  CE2Error_t 
*  - CE2_OK - on success
*  - Otherwise - error code:
*
* \brief \b 
* Description:
*  This function is used to perform C2 One Way operation on 64-bit block 
*  of data. The function executes the following major steps:
*  1. Executing the C2 ECB encrypt operation of Data2 with key = Data1 
*  by calling the low level function LLF_C2_ECB_Encrypt.
*  2. Performs XOR of encrypt output with Data2 and output this result.
*
*  \b 
* Algorithm:
*  -# Load the secret constant if need;
*  -# Executing the C2 ECB encrypt operation of Data2 with key = Data1;
*  -# Performs XOR of encrypt output with Data2 and output this result.
***************************************************************/
CE2Error_t  LLF_C2_OneWayFunc(CE2_C2Cipher_Key_t   Data1_ptr,     
                              CE2_C2_BLOCK_BYTES_t Data2_ptr,
                              CE2_C2_BLOCK_BYTES_t DataOut_ptr);   

#ifdef __cplusplus
}
#endif

#endif /* LLF_C2_H */
